/* Blink Example

   This example code is in the Public Domain (or CC0 licensed, at your option.)

   Unless required by applicable law or agreed to in writing, this
   software is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
   CONDITIONS OF ANY KIND, either express or implied.
*/
#include <stdio.h>
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "driver/gpio.h"
#include "driver/i2c.h"
#include "esp_log.h"
#include "sdkconfig.h"

#define I2C_MASTER_SCL_IO 10
#define I2C_MASTER_SDA_IO 9
#define I2C_MASTER_NUM I2C_NUM_0
#define I2C_MASTER_FREQ_HZ 100000
#define I2C_MASTER_TX_BUF_DISABLE 0
#define I2C_MASTER_RX_BUF_DISABLE 0
#define I2C_MASTER_TIMEOUT_MS 1000

// BH1750 定义
#define BH1750_I2C_ADDR_LOW 0x23  // ADDR引脚接地或悬空时的地址
#define BH1750_I2C_ADDR_HIGH 0x5C // ADDR引脚接VCC时的地址

// BH1750 指令
#define BH1750_POWER_DOWN 0x00 // 断电
#define BH1750_POWER_ON 0x01   // 通电等待指令
#define BH1750_RESET 0x07      // 重置数据寄存器

// BH1750 连续测量模式
#define BH1750_CONT_HIGH_RES 0x10  // 连续高分辨率模式（1lx分辨率，120ms测量时间）
#define BH1750_CONT_HIGH_RES2 0x11 // 连续高分辨率模式2（0.5lx分辨率，120ms测量时间）
#define BH1750_CONT_LOW_RES 0x13   // 连续低分辨率模式（4lx分辨率，16ms测量时间）

// BH1750 单次测量模式
#define BH1750_ONE_HIGH_RES 0x20  // 单次高分辨率模式（1lx分辨率，120ms测量时间）
#define BH1750_ONE_HIGH_RES2 0x21 // 单次高分辨率模式2（0.5lx分辨率，120ms测量时间）
#define BH1750_ONE_LOW_RES 0x23   // 单次低分辨率模式（4lx分辨率，16ms测量时间）

static const char *TAG = "example";

/* GPIO pin for LED */
#define BLINK_GPIO GPIO_NUM_15

/* BH1750 sensor address */
static uint8_t bh1750_addr = BH1750_I2C_ADDR_LOW;

static uint8_t s_led_state = 0;

TaskHandle_t led_task_handle = NULL;

static void blink_led_task(void *pvParameter)
{
    /* Toggle the GPIO level */
    while (1)
    {
        ESP_LOGI(TAG, "Turning the LED %s!", s_led_state ? "ON" : "OFF");
        s_led_state = !s_led_state;
        gpio_set_level(BLINK_GPIO, s_led_state);
        vTaskDelay(pdMS_TO_TICKS(1000)); // 1 second delay
    }

    /* Task should never reach here, but if it does, delete itself */
    vTaskDelete(NULL);
}

static void configure_led(void)
{
    ESP_LOGI(TAG, "Configuring GPIO15 as output with pull-up!");

    /* Reset the GPIO pin */
    gpio_reset_pin(BLINK_GPIO);

    /* Configure GPIO as output with pull-up */
    gpio_config_t io_conf = {
        .intr_type = GPIO_INTR_DISABLE,
        .mode = GPIO_MODE_OUTPUT,
        .pin_bit_mask = (1ULL << BLINK_GPIO),
        .pull_down_en = GPIO_PULLDOWN_DISABLE,
        .pull_up_en = GPIO_PULLUP_ENABLE,
    };
    gpio_config(&io_conf);

    /* Set initial state to LOW */
    gpio_set_level(BLINK_GPIO, 0);
}

/**
 * @brief i2c master initialization
 */
static esp_err_t i2c_master_init(void)
{
    int i2c_master_port = I2C_MASTER_NUM;

    i2c_config_t conf = {
        .mode = I2C_MODE_MASTER,
        .sda_io_num = I2C_MASTER_SDA_IO,
        .scl_io_num = I2C_MASTER_SCL_IO,
        .sda_pullup_en = GPIO_PULLUP_ENABLE,
        .scl_pullup_en = GPIO_PULLUP_ENABLE,
        .master.clk_speed = I2C_MASTER_FREQ_HZ,
    };

    i2c_param_config(i2c_master_port, &conf);

    return i2c_driver_install(i2c_master_port, conf.mode, I2C_MASTER_RX_BUF_DISABLE, I2C_MASTER_TX_BUF_DISABLE, 0);
}

/**
 * @brief I2C scanner function
 */
static void i2c_scanner(void)
{
    ESP_LOGI(TAG, "Starting I2C scan...");

    uint8_t address;
    esp_err_t ret;
    int devices_found = 0;

    for (address = 1; address < 127; address++)
    {
        i2c_cmd_handle_t cmd = i2c_cmd_link_create();
        i2c_master_start(cmd);
        i2c_master_write_byte(cmd, (address << 1) | I2C_MASTER_WRITE, true);
        i2c_master_stop(cmd);

        ret = i2c_master_cmd_begin(I2C_MASTER_NUM, cmd, 10 / portTICK_PERIOD_MS);
        i2c_cmd_link_delete(cmd);

        if (ret == ESP_OK)
        {
            ESP_LOGI(TAG, "Found device at address: 0x%02X", address);
            devices_found++;
        }
    }

    if (devices_found == 0)
    {
        ESP_LOGI(TAG, "No I2C devices found!");
    }
    else
    {
        ESP_LOGI(TAG, "Scan completed. Found %d device(s).", devices_found);
    }
}

/**
 * @brief 向BH1750发送指令
 */
static esp_err_t bh1750_write_cmd(uint8_t cmd)
{
    i2c_cmd_handle_t i2c_cmd = i2c_cmd_link_create();
    i2c_master_start(i2c_cmd);
    i2c_master_write_byte(i2c_cmd, (bh1750_addr << 1) | I2C_MASTER_WRITE, true);
    i2c_master_write_byte(i2c_cmd, cmd, true);
    i2c_master_stop(i2c_cmd);

    esp_err_t ret = i2c_master_cmd_begin(I2C_MASTER_NUM, i2c_cmd, I2C_MASTER_TIMEOUT_MS / portTICK_PERIOD_MS);
    i2c_cmd_link_delete(i2c_cmd);

    return ret;
}

/**
 * @brief 从BH1750读取数据
 */
static esp_err_t bh1750_read_data(uint8_t *data, size_t len)
{
    i2c_cmd_handle_t i2c_cmd = i2c_cmd_link_create();
    i2c_master_start(i2c_cmd);
    i2c_master_write_byte(i2c_cmd, (bh1750_addr << 1) | I2C_MASTER_READ, true);

    if (len > 1)
    {
        i2c_master_read(i2c_cmd, data, len - 1, I2C_MASTER_ACK);
    }
    i2c_master_read_byte(i2c_cmd, data + len - 1, I2C_MASTER_NACK);
    i2c_master_stop(i2c_cmd);

    esp_err_t ret = i2c_master_cmd_begin(I2C_MASTER_NUM, i2c_cmd, I2C_MASTER_TIMEOUT_MS / portTICK_PERIOD_MS);
    i2c_cmd_link_delete(i2c_cmd);

    return ret;
}

/**
 * @brief Initialize BH1750 sensor
 */
static esp_err_t bh1750_init_sensor(void)
{
    esp_err_t ret;

    // 先断电再上电，复位传感器
    ret = bh1750_write_cmd(BH1750_POWER_DOWN);
    if (ret != ESP_OK)
    {
        ESP_LOGE(TAG, "Failed to power down BH1750: %s", esp_err_to_name(ret));
        return ret;
    }

    vTaskDelay(pdMS_TO_TICKS(10)); // 等待10ms

    // 上电
    ret = bh1750_write_cmd(BH1750_POWER_ON);
    if (ret != ESP_OK)
    {
        ESP_LOGE(TAG, "Failed to power on BH1750: %s", esp_err_to_name(ret));
        return ret;
    }

    vTaskDelay(pdMS_TO_TICKS(10)); // 等待10ms

    // 重置数据寄存器
    ret = bh1750_write_cmd(BH1750_RESET);
    if (ret != ESP_OK)
    {
        ESP_LOGE(TAG, "Failed to reset BH1750: %s", esp_err_to_name(ret));
        return ret;
    }

    vTaskDelay(pdMS_TO_TICKS(10)); // 等待10ms

    // 设置为连续高分辨率模式
    ret = bh1750_write_cmd(BH1750_CONT_HIGH_RES);
    if (ret != ESP_OK)
    {
        ESP_LOGE(TAG, "Failed to set BH1750 mode: %s", esp_err_to_name(ret));
        return ret;
    }

    ESP_LOGI(TAG, "BH1750 sensor initialized successfully");
    return ESP_OK;
}

/**
 * @brief Read light level from BH1750 sensor
 */
static esp_err_t bh1750_read_light(uint16_t *lux)
{
    uint8_t data[2];
    esp_err_t ret = bh1750_read_data(data, 2);

    if (ret != ESP_OK)
    {
        ESP_LOGE(TAG, "Failed to read from BH1750: %s", esp_err_to_name(ret));
        return ret;
    }

    // BH1750返回的是16位大端序数据
    // 高字节在前，低字节在后
    uint16_t raw_data = (data[0] << 8) | data[1];

    // 在高分辨率模式下，每个计数值等于1.2lux
    // 转换公式: lux = raw_data / 1.2
    // 为了避免浮点运算，使用整数运算: lux = raw_data * 10 / 12
    *lux = (raw_data * 10) / 12;

    return ESP_OK;
}

/**
 * @brief BH1750 monitoring task
 */
static void bh1750_task(void *pvParameter)
{
    uint16_t lux;

    while (1)
    {
        if (bh1750_read_light(&lux) == ESP_OK)
        {
            ESP_LOGI(TAG, "Light level: %d lux", lux);
        }

        vTaskDelay(pdMS_TO_TICKS(2000)); // Read every 2 seconds
    }

    vTaskDelete(NULL);
}

void app_main(void)
{
    configure_led();

    // Initialize I2C master
    ESP_ERROR_CHECK(i2c_master_init());
    ESP_LOGI(TAG, "I2C initialized successfully");

    // Perform I2C scan
    i2c_scanner();

    // Initialize BH1750 sensor
    esp_err_t ret = bh1750_init_sensor();
    if (ret == ESP_OK)
    {
        // Create BH1750 monitoring task
        xTaskCreate(bh1750_task, "bh1750_task", 4096, NULL, 4, NULL);
        ESP_LOGI(TAG, "BH1750 monitoring task created successfully");
    }
    else
    {
        ESP_LOGE(TAG, "Failed to initialize BH1750 sensor, skipping light monitoring");
    }

    xTaskCreate(blink_led_task, "blink_led", 2048, NULL, 5, &led_task_handle);

    ESP_LOGI(TAG, "LED blink task created successfully");
}
