/* Firebeetle 2 ESP32-C5 6通道遥控器
 *     
 * 2~5 -> 模拟输入（1-4通道摇杆），15、26~28 -> 4P拨码开关（通道正反向）
 * 6 -> 电压检测，7~8 -> 5-6通道两段扭子开关
 * 12 -> 有源蜂鸣器
 * 
 * NRF24L01 | Firebeetle 2 ESP32-C5
 * CE    -> 9
 * CSN   -> 10
 * MOSI  -> 24
 * MISO  -> 25
 * SCK   -> 23
 * IRQ   -> 无连接
 * VCC   -> 3.3V
 * GND   -> GND
 */
#include <SPI.h>
#include <nRF24L01.h>
#include <RF24.h>

// 无线通信地址（需与接收端一致）
const uint64_t pipeOut = 0x123456789;  
// 死区值（摇杆中位附近±deadzone范围视为中位，避免抖动）
const int deadzone = 5;              
// NRF24L01初始化（CE=9，CSN=10）
RF24 radio(9, 10);                     

// 发送重试配置
const int maxRetries = 3;       // 最大重试次数
const int retryDelay = 10;      // 重试间隔（毫秒）

// 控制信号结构体（修正数据类型为int，支持负数和大数值）
struct Signal {
  int ele;   // 1通道：升降舵（-512~512）
  int ail;   // 2通道：副翼（-512~512）
  int thr;   // 3通道：油门（-512~512）
  int rud;   // 4通道：方向舵（-512~512）
  byte aux1; // 5通道：两段扭子开关（-512/512映射后）
  byte aux2; // 6通道：两段扭子开关（-512/512映射后）
};
Signal data;

// 初始化通道值（安全状态）
void ResetData() {
  data.ele = 0;     // 升降舵中位（死区中心）
  data.ail = 0;     // 副翼中位
  data.thr = 0;     // 油门中位
  data.rud = 0;     // 方向舵中位
  data.aux1 = -512; // 5通道默认值
  data.aux2 = -512; // 6通道默认值
}

// 死区处理函数（输入值在±deadzone范围内时，强制设为中心值）
int applyDeadzone(int value, int deadzone, int center) {
  if (abs(value - center) <= deadzone) {
    return center; // 落入死区，返回中心值
  }
  return value;    // 超出死区，返回原始值
}

// 带重试机制的发送函数
bool sendData(const void* data, size_t dataSize) {
  for (int retryCount = 0; retryCount < maxRetries; retryCount++) {
    if (radio.write(data, dataSize)) {
      // 发送成功
      Serial.print("发送成功，重试次数：");
      Serial.println(retryCount);
      return true;
    }
    // 发送失败，延时后重试
    delay(retryDelay);
  }
  // 达到最大重试次数仍失败
  Serial.println("发送失败，已达到最大重试次数");
  return false;
}

void setup() {
  radio.begin();
  radio.openWritingPipe(pipeOut);  // 设置发送地址
  radio.stopListening();           // 切换为发射模式
  // 优化NRF24L01传输参数（提升抗干扰能力）
  radio.setDataRate(RF24_250KBPS); // 低速率模式，抗干扰更强
  radio.setPALevel(RF24_PA_HIGH);  // 高功率模式（确保电源稳定供电）
  radio.setRetries(5, 10);        // 模块硬件重试：5次间隔10*250=2500μs
  ResetData();                     // 初始化通道数据
  
  Serial.begin(115200);
  analogReadResolution(10);        // ADC设为10位（0~1023）
  
  // 配置引脚模式（拨码开关启用内部上拉，避免悬空）
  pinMode(15, INPUT_PULLUP);   // 1通道方向拨码
  pinMode(26, INPUT_PULLUP);   // 2通道方向拨码
  pinMode(27, INPUT_PULLUP);   // 3通道方向拨码
  pinMode(28, INPUT_PULLUP);   // 4通道方向拨码
  pinMode(7, INPUT_PULLUP);    // 5通道扭子开关（上拉，OFF=0）
  pinMode(8, INPUT_PULLUP);    // 6通道扭子开关（上拉，OFF=0）
  pinMode(12, OUTPUT);         // 有源蜂鸣器控制
  
  // 系统启动提示（蜂鸣3声）
  for(int i=0; i<3; i++){
    digitalWrite(12, HIGH);
    delay(100);
    digitalWrite(12, LOW);
    delay(100);
  }
}

void loop() {
  // 1. 读取1-4通道模拟信号并映射到目标范围
  int raw_ele = map(analogRead(2), 2, 840, -512, 512);  // 升降舵原始值
  int raw_ail = map(analogRead(3), 2, 838, -512, 512);  // 副翼原始值
  int raw_thr = map(analogRead(4), 0, 839, -512, 512);  // 油门原始值
  int raw_rud = map(analogRead(5), 2, 838, -512, 512);  // 方向舵原始值
  
  // 2. 应用死区处理（所有带中位的通道）
  data.ele = applyDeadzone(raw_ele, deadzone, 0);
  data.ail = applyDeadzone(raw_ail, deadzone, 0);
  data.rud = applyDeadzone(raw_rud, deadzone, 0);
  data.thr = applyDeadzone(raw_thr, deadzone, 0);
  
  // 3. 读取5-6通道扭子开关信号并映射
  data.aux1 = digitalRead(7) ? 512 : -512;
  data.aux2 = digitalRead(8) ? 512 : -512;
  
  // 4. 通道正反向切换（拨码开关ON时反向，ON=LOW）
  data.ele = digitalRead(15) ? data.ele : -data.ele;
  data.ail = digitalRead(26) ? data.ail : -data.ail;
  data.thr = digitalRead(27) ? data.thr : -data.thr;
  data.rud = digitalRead(28) ? data.rud : -data.rud;
  
  // 5. 带重试机制的发送数据
  sendData(&data, sizeof(Signal));
  
  // 调试输出通道数据
  Serial.print("ele:"); Serial.print(data.ele);
  Serial.print(" ail:"); Serial.print(data.ail);
  Serial.print(" thr:"); Serial.print(data.thr);
  Serial.print(" rud:"); Serial.print(data.rud);
  Serial.print(" aux1:"); Serial.print(data.aux1);
  Serial.print(" aux2:"); Serial.println(data.aux2);
  
  delay(20);  // 基础发送间隔，确保总频率稳定
}