#include <Arduino.h>
#include <Wire.h>
#include "Adafruit_SHT31.h"
#include <Adafruit_BMP085.h>
#include <WiFi.h>
#include "Adafruit_MQTT.h"
#include "Adafruit_MQTT_Client.h"

// ============================
// 系统配置参数
// ============================
#define DEBUG_MODE              1     // 调试模式：0-关闭 1-打开
#define ENABLE_SLEEP            1     // 允许休眠
#define ENABLE_MQTT             1     // 启用MQTT功能

#define uS_TO_S_FACTOR         1000000ULL  // 微秒到秒转换因子
#if ( DEBUG_MODE == 1 )
#define TIME_TO_SLEEP          10     // ESP32休眠时间（秒）
#else
#define TIME_TO_SLEEP          60     // ESP32休眠时间（秒）
#endif

// BOOT按钮配置
#define BUTTON_BOOT             28    // BOOT按钮引脚
#define LONG_PRESS_TIME         3000  // 长按时间（毫秒）

// ============================
// 电池检测配置
// ============================
#define BATTERY_PIN             1       // 电池检测引脚（GPIO1）
#define BATTERY_MAX_VOLTAGE     4.2     // 锂电池满电电压
#define BATTERY_MIN_VOLTAGE     3.0     // 锂电池最低电压
#define VOLTAGE_DIVIDER_RATIO   2.0     // 分压电阻比例 (R1+R2)/R2
#define BATTERY_ADC_SAMPLES     16      // ADC采样次数

RTC_DATA_ATTR int bootCount = 0;      // 启动计数（RTC保持）
RTC_DATA_ATTR bool skipSleep = false; // 跳过休眠标志

// ============================
// WiFi配置
// ============================
#define WLAN_SSID       "WiFi名称"
#define WLAN_PASS       "WiFi密码"

// ============================
// MQTT配置
// ============================
#define AIO_SERVER      "192.168.1.185"  // MQTT服务器地址
#define AIO_SERVERPORT  1883             // MQTT端口
#define AIO_USERNAME    "siot"           // MQTT用户名
#define AIO_KEY         "dfrobot"        // MQTT密码

// 气压传感器配置
#define BMP180_REFERENCE_PRESSURE  102500    // 参考海平面气压值

// ============================
// 光线传感器配置
// ============================
#define LDR_ADC_PIN            2      // 光线传感器引脚（GPIO2）
#define VOLTAGE_DIVIDER_FACTOR 1.5    // 分压电路系数

// PT550传感器参数
const float LUX_MAX = 6000.0;         // 最大流明值
const float VOLTAGE_MAX = 5.0;        // 最大输出电压

// 光线强度档位描述
const char* LIGHT_LEVELS[] = {
  "完全黑暗",    // 档位0: 0-10 Lux
  "极弱光线",    // 档位1: 10-50 Lux
  "很弱光线",    // 档位2: 50-200 Lux
  "弱光线",      // 档位3: 200-500 Lux
  "中等偏弱",    // 档位4: 500-1000 Lux
  "中等光线",    // 档位5: 1000-1500 Lux
  "中等偏强",    // 档位6: 1500-2500 Lux
  "强光线",      // 档位7: 2500-3500 Lux
  "很强光线",    // 档位8: 3500-4500 Lux
  "刺眼光线"     // 档位9: 4500+ Lux
};

// ============================
// 传感器对象声明
// ============================
Adafruit_SHT31 sht31 = Adafruit_SHT31();  // SHT31温湿度传感器
Adafruit_BMP085 bmp;                      // BMP180气压传感器


// ============================
// WiFi对象声明
// ============================
WiFiClient client;

// ============================
// MQTT对象声明
// ============================
Adafruit_MQTT_Client mqtt(&client, AIO_SERVER, AIO_SERVERPORT, AIO_USERNAME, AIO_KEY);

// MQTT发布对象
Adafruit_MQTT_Publish sht30_temp = Adafruit_MQTT_Publish(&mqtt, "dapeng_01/sht30_temp");
Adafruit_MQTT_Publish sht30_humi = Adafruit_MQTT_Publish(&mqtt, "dapeng_01/sht30_humi");
Adafruit_MQTT_Publish bmp180_temp = Adafruit_MQTT_Publish(&mqtt, "dapeng_01/bmp180_temp");
Adafruit_MQTT_Publish bmp180_pres = Adafruit_MQTT_Publish(&mqtt, "dapeng_01/bmp180_pres");
Adafruit_MQTT_Publish bmp180_alti = Adafruit_MQTT_Publish(&mqtt, "dapeng_01/bmp180_alti");
Adafruit_MQTT_Publish light_lux = Adafruit_MQTT_Publish(&mqtt, "dapeng_01/light_lux");
Adafruit_MQTT_Publish light_level = Adafruit_MQTT_Publish(&mqtt, "dapeng_01/light_level");
Adafruit_MQTT_Publish light_desc = Adafruit_MQTT_Publish(&mqtt, "dapeng_01/light_desc");
Adafruit_MQTT_Publish boot_count = Adafruit_MQTT_Publish(&mqtt, "dapeng_01/boot_count");
Adafruit_MQTT_Publish battery_voltage = Adafruit_MQTT_Publish(&mqtt, "dapeng_01/battery_voltage");
Adafruit_MQTT_Publish battery_percentage = Adafruit_MQTT_Publish(&mqtt, "dapeng_01/battery_percentage");
Adafruit_MQTT_Publish battery_status = Adafruit_MQTT_Publish(&mqtt, "dapeng_01/battery_status");

// SHT31加热器控制变量
bool enableHeater = false;                // 加热器状态
uint8_t loopCnt = 0;                      // 循环计数器

// 按钮状态变量
unsigned long buttonPressTime = 0;        // 按钮按下时间
bool buttonActive = false;                // 按钮激活状态

// 传感器数据存储
struct SensorData {
  float sht30_temperature;
  float sht30_humidity;
  float bmp180_temperature;
  float bmp180_pressure;
  float bmp180_altitude;
  float light_lux;
  int light_level;
  const char* light_description;
  float battery_voltage;        // 电池电压
  int battery_percentage;       // 电池电量百分比
  const char* battery_status;   // 电池状态描述
} sensorData;

// ============================
// 函数声明
// ============================
float readLuxValue(int samples = 8);              // 读取光线强度
const char* convertLuxToLightLevel(float lux);    // 流明值转光线描述
int getLightLevel(float lux);                     // 获取光线档位
void i2c_scan();                                  // I2C设备扫描
void printSensorData();                           // 打印传感器数据
bool initSensors();                               // 初始化传感器
void readAllSensors();                            // 读取所有传感器数据
void connectWiFi();                               // 连接WiFi
void MQTT_connect();                              // 连接MQTT服务器
bool publishSensorData();                         // 发布传感器数据到MQTT
void checkButton();                               // 检查按钮状态
void handleButtonPress();                         // 处理按钮按下
void setupWakeupSources();                        // 设置唤醒源
String formatFloat(float value, int decimals);    // 格式化浮点数为字符串
float readBatteryVoltage();                       // 读取电池电压
int calculateBatteryPercentage(float voltage);    // 计算电池电量百分比
const char* getBatteryStatus(int percentage);     // 获取电池状态描述
void readBatteryData();                           // 读取电池数据

// ============================
// 系统初始化
// ============================
void setup() {
  Serial.begin(115200);
  delay(1000);

  Serial.println("系统启动中...");

  // 配置BOOT按钮引脚
  pinMode(BUTTON_BOOT, INPUT_PULLUP);
  
  // 检查唤醒原因
  esp_sleep_wakeup_cause_t wakeup_reason = esp_sleep_get_wakeup_cause();
  switch(wakeup_reason) {
    case ESP_SLEEP_WAKEUP_TIMER:
      Serial.println("⏰ 由定时器唤醒");
      break;
    case ESP_SLEEP_WAKEUP_EXT0:
      Serial.println("🔔 由外部信号唤醒");
      break;
    case ESP_SLEEP_WAKEUP_EXT1:
      Serial.println("🔔 由外部信号唤醒");
      break;
    case ESP_SLEEP_WAKEUP_GPIO:
      Serial.println("🔔 由GPIO唤醒");
      break;
    default:
      Serial.println("🚀 正常启动或复位");
      break;
  }

  ++bootCount;
  Serial.println("启动次数: " + String(bootCount));

  // 连接WiFi
#if (ENABLE_MQTT == 1)
  connectWiFi();
#endif

  // 初始化I2C总线（SDA: GPIO9, SCL: GPIO10）
  Wire.begin(9, 10);

  // 扫描I2C设备
  i2c_scan();

  // 初始化所有传感器
  if (!initSensors()) {
    Serial.println("传感器初始化失败，系统暂停！");
    while (1) delay(1000);
  }

  // 设置ADC分辨率（12位：0-4095）
  analogReadResolution(12);
  analogSetAttenuation(ADC_11db); // 设置11dB衰减 (0-3.9V范围)

  // 配置唤醒源
  setupWakeupSources();
  
  Serial.println("系统初始化完成，开始监测环境数据...\n");

#if (ENABLE_SLEEP == 1)
  sht31.heater(true);
#endif
}

// ============================
// 主循环
// ============================
void loop() {
  // 检查按钮状态
  checkButton();
  
  // 读取所有传感器数据
  readAllSensors();
  
  // 打印传感器数据到串口
  printSensorData();

  // 发布数据到MQTT
#if (ENABLE_MQTT == 1)
  if (publishSensorData()) {
    Serial.println("✅ 数据成功发布到MQTT");
  } else {
    Serial.println("❌ MQTT数据发布失败");
  }
#endif

  // 每10次循环切换加热器状态（约20秒间隔）
  if (loopCnt >= 10) {
    enableHeater = !enableHeater;
    sht31.heater(enableHeater);
    
    if (DEBUG_MODE) {
      Serial.print("SHT31加热器状态: ");
      Serial.println(enableHeater ? "启用" : "关闭");
    }
    loopCnt = 0;
  }

  loopCnt++;

#if (ENABLE_SLEEP == 1)
  if (loopCnt >= 3 && !skipSleep) {
    Serial.println("进入休眠状态，" + String(TIME_TO_SLEEP) + "秒后自动唤醒");
    Serial.println("长按BOOT按钮可取消休眠");
    Serial.flush();

#if (ENABLE_MQTT == 1)
    delay(2000);

    // 断开MQTT连接
    if (mqtt.connected()) {
      mqtt.disconnect();
    }
    // 断开WiFi以节省功耗
    WiFi.disconnect(true);
    WiFi.mode(WIFI_OFF);
#endif

    // 进入休眠
    esp_sleep_enable_timer_wakeup(TIME_TO_SLEEP * uS_TO_S_FACTOR);
    esp_deep_sleep_start();
  }
#endif

  delay(2000);  // 2秒采样间隔
}

// ============================
// 电池检测函数实现
// ============================

/**
 * @brief 读取电池电压
 * @return float 电池电压值（伏特）
 */
float readBatteryVoltage() {
  int raw = 0;
  int raw_sum = 0;
  for (int i = 0; i < BATTERY_ADC_SAMPLES; i++) {
    raw = analogRead(BATTERY_PIN);
    #if ( DEBUG_MODE == 1 )
      Serial.printf("电池电量检测第%d次读取数值：%d\n", i+1, raw);
    #endif
    raw_sum += raw;
    delay(1);
  }
  raw = raw_sum / BATTERY_ADC_SAMPLES; // 计算平均值
  #if ( DEBUG_MODE == 1 )
  Serial.printf("电池电量检测%d次平均数值：%d\n", BATTERY_ADC_SAMPLES, raw);
  #endif

  // 将ADC值转换为电压 (12位ADC，参考电压3.3V)
  float voltage = (raw * BATTERY_MAX_VOLTAGE) / 4095.0; // 单位V
  return voltage * VOLTAGE_DIVIDER_RATIO; // 计算实际电池电压
}

/**
 * @brief 计算电池电量百分比
 * @param voltage 电池电压
 * @return int 电量百分比（0-100）
 */
int calculateBatteryPercentage(float voltage) {
  // 电压超出范围处理
  if (voltage >= BATTERY_MAX_VOLTAGE) return 100;
  if (voltage <= BATTERY_MIN_VOLTAGE) return 0;
  
  // 使用非线性曲线计算电量百分比（更准确）
  // 锂电池放电曲线不是线性的
  float percentage = 0;
  if (voltage > 3.9) {
    percentage = 100 - (4.2 - voltage) * 60;
  } else if (voltage > 3.7) {
    percentage = 60 - (3.9 - voltage) * 100;
  } else if (voltage > 3.5) {
    percentage = 40 - (3.7 - voltage) * 100;
  } else {
    percentage = 10 - (3.5 - voltage) * 20;
  }
  
  // 确保百分比在0-100范围内
  return constrain(percentage, 0, 100);
}

/**
 * @brief 获取电池状态描述
 * @param percentage 电量百分比
 * @return const char* 状态描述
 */
const char* getBatteryStatus(int percentage) {
  if (percentage >= 80) return "电量充足";
  else if (percentage >= 50) return "电量中等";
  else if (percentage >= 20) return "电量偏低";
  else return "电量不足请充电";
}

/**
 * @brief 读取电池数据
 */
void readBatteryData() {
  sensorData.battery_voltage = readBatteryVoltage();
  sensorData.battery_percentage = calculateBatteryPercentage(sensorData.battery_voltage);
  sensorData.battery_status = getBatteryStatus(sensorData.battery_percentage);
}

// ============================
// 格式化函数
// ============================

/**
 * @brief 格式化浮点数为字符串
 * @param value 浮点数值
 * @param decimals 小数位数
 * @return String 格式化后的字符串
 */
String formatFloat(float value, int decimals) {
  char buffer[20];
  dtostrf(value, 1, decimals, buffer);
  return String(buffer);
}

// ============================
// 唤醒源配置函数
// ============================

/**
 * @brief 设置唤醒源
 */
void setupWakeupSources() {
  // ESP32-C5使用GPIO唤醒而不是ext0/ext1
  // 配置BOOT按钮为唤醒源
  gpio_wakeup_enable((gpio_num_t)BUTTON_BOOT, GPIO_INTR_LOW_LEVEL);
  esp_sleep_enable_gpio_wakeup();
  
  Serial.println("✅ BOOT按钮已配置为唤醒源");
}

// ============================
// 按钮处理函数
// ============================

/**
 * @brief 检查按钮状态
 */
void checkButton() {
  // 读取按钮状态
  if (digitalRead(BUTTON_BOOT) == LOW) {
    if (buttonActive == false) {
      buttonActive = true;
      buttonPressTime = millis();
    }
    // 检查是否长按
    if (millis() - buttonPressTime > LONG_PRESS_TIME) {
      handleButtonPress();
    }
  } else {
    buttonActive = false;
  }
}

/**
 * @brief 处理按钮长按
 */
void handleButtonPress() {
  if (skipSleep) {
    // 如果已经设置了跳过休眠，再次长按取消
    skipSleep = false;
    Serial.println("🔄 休眠模式已启用，下次将正常休眠");
  } else {
    // 设置跳过休眠标志
    skipSleep = true;
    Serial.println("⏸️ 休眠已取消，长按BOOT按钮恢复休眠");
    // 重置循环计数，延长工作时间
    loopCnt = 0;
  }
  
  // 等待按钮释放
  while (digitalRead(BUTTON_BOOT) == LOW) {
    delay(100);
  }
  
  // 重置按钮状态
  buttonActive = false;
  buttonPressTime = 0;
}

// ============================
// WiFi和MQTT函数实现
// ============================

/**
 * @brief 连接WiFi网络
 */
void connectWiFi() {
  Serial.println();
  Serial.println("正在连接WiFi...");
  Serial.print("SSID: ");
  Serial.println(WLAN_SSID);

  WiFi.begin(WLAN_SSID, WLAN_PASS);
  
  int attempts = 0;
  while (WiFi.status() != WL_CONNECTED && attempts < 20) {
    delay(500);
    Serial.print(".");
    attempts++;
  }
  
  if (WiFi.status() == WL_CONNECTED) {
    Serial.println();
    Serial.println("✅ WiFi连接成功!");
    Serial.print("IP地址: ");
    Serial.println(WiFi.localIP());
  } else {
    Serial.println();
    Serial.println("❌ WiFi连接失败!");
  }
  Serial.println();
}

/**
 * @brief 连接MQTT服务器
 */
void MQTT_connect() {
  int8_t ret;

  // 如果已经连接，直接返回
  if (mqtt.connected()) {
    return;
  }

  Serial.print("正在连接MQTT服务器...");

  uint8_t retries = 3;
  while ((ret = mqtt.connect()) != 0) {
    Serial.println(mqtt.connectErrorString(ret));
    Serial.println("5秒后重试MQTT连接...");
    mqtt.disconnect();
    delay(5000);
    retries--;
    if (retries == 0) {
      Serial.println("MQTT连接失败，跳过本次发布");
      return;
    }
  }
  Serial.println("✅ MQTT连接成功!");
}

/**
 * @brief 发布传感器数据到MQTT
 * @return bool 发布是否成功
 */
bool publishSensorData() {
  // 确保MQTT连接
  MQTT_connect();
  
  if (!mqtt.connected()) {
    return false;
  }

  bool success = true;
  
  // 格式化所有数据为字符串，保持与Serial输出一致的格式
  String tempSHT = formatFloat(sensorData.sht30_temperature, 1) + "℃";
  String humiSHT = formatFloat(sensorData.sht30_humidity, 0) + "%";
  String tempBMP = formatFloat(sensorData.bmp180_temperature, 1) + "℃";
  String presBMP = formatFloat(sensorData.bmp180_pressure / 1000.0, 3) + "kPa";
  String altiBMP = formatFloat(sensorData.bmp180_altitude, 1) + "米";
  String luxStr = formatFloat(sensorData.light_lux, 0) + "Lux";
  String levelStr = String(sensorData.light_level);
  String descStr = String(sensorData.light_description);
  
  // 电池数据格式化
  String batVoltStr = formatFloat(sensorData.battery_voltage, 2) + "V";
  String batPctStr = String(sensorData.battery_percentage) + "%";
  String batStatusStr = String(sensorData.battery_status);
  
  // 发布SHT30数据 - 使用字符串格式
  if (!sht30_temp.publish(tempSHT.c_str())) {
    Serial.println("SHT30温度发布失败");
    success = false;
  }
  if (!sht30_humi.publish(humiSHT.c_str())) {
    Serial.println("SHT30湿度发布失败");
    success = false;
  }
  
  // 发布BMP180数据
  if (!bmp180_temp.publish(tempBMP.c_str())) {
    Serial.println("BMP180温度发布失败");
    success = false;
  }
  if (!bmp180_pres.publish(presBMP.c_str())) {
    Serial.println("BMP180气压发布失败");
    success = false;
  }
  if (!bmp180_alti.publish(altiBMP.c_str())) {
    Serial.println("BMP180海拔发布失败");
    success = false;
  }
  
  // 发布光线数据
  if (!light_lux.publish(luxStr.c_str())) {
    Serial.println("光线流明值发布失败");
    success = false;
  }
  if (!light_level.publish(levelStr.c_str())) {
    Serial.println("光线档位发布失败");
    success = false;
  }
  if (!light_desc.publish(descStr.c_str())) {
    Serial.println("光线描述发布失败");
    success = false;
  }
  
  // 发布电池数据
  if (!battery_voltage.publish(batVoltStr.c_str())) {
    Serial.println("电池电压发布失败");
    success = false;
  }
  if (!battery_percentage.publish(batPctStr.c_str())) {
    Serial.println("电池电量发布失败");
    success = false;
  }
  if (!battery_status.publish(batStatusStr.c_str())) {
    Serial.println("电池状态发布失败");
    success = false;
  }
  
  if (!boot_count.publish(String(bootCount).c_str())) {
    Serial.println("启动次数发布失败");
    success = false;
  }
  return success;
}

// ============================
// 传感器函数实现
// ============================

/**
 * @brief 初始化所有传感器
 * @return bool 初始化是否成功
 */
bool initSensors() {
  bool success = true;
  
  // 初始化SHT31温湿度传感器
  Serial.println("初始化SHT31温湿度传感器...");
  if (!sht31.begin(0x44)) {
    Serial.println("错误：未找到SHT31传感器！");
    success = false;
  } else {
    Serial.println("SHT31初始化成功");
    Serial.print("加热器初始状态: ");
    Serial.println(sht31.isHeaterEnabled() ? "启用" : "关闭");
  }

  // 初始化BMP180气压传感器
  Serial.println("初始化BMP180气压传感器...");
  if (!bmp.begin()) {
    Serial.println("错误：未找到BMP180传感器！");
    success = false;
  } else {
    Serial.println("BMP180初始化成功");
  }

  return success;
}

/**
 * @brief 读取所有传感器数据
 */
void readAllSensors() {
  // 读取SHT31温湿度数据
  sensorData.sht30_temperature = sht31.readTemperature();
  sensorData.sht30_humidity = sht31.readHumidity();
  
  // 读取BMP180数据
  sensorData.bmp180_temperature = bmp.readTemperature();
  sensorData.bmp180_pressure = bmp.readPressure();
  sensorData.bmp180_altitude = bmp.readAltitude(BMP180_REFERENCE_PRESSURE);  // 参考海平面压力

  // 读取光线数据
  sensorData.light_lux = readLuxValue(8);
  sensorData.light_description = convertLuxToLightLevel(sensorData.light_lux);
  sensorData.light_level = getLightLevel(sensorData.light_lux);

  // 读取电池数据
  readBatteryData();

  // 数据有效性检查
  if (isnan(sensorData.sht30_temperature)) sensorData.sht30_temperature = 0;
  if (isnan(sensorData.sht30_humidity)) sensorData.sht30_humidity = 0;
  if (isnan(sensorData.bmp180_temperature)) sensorData.bmp180_temperature = 0;
  if (isnan(sensorData.bmp180_pressure)) sensorData.bmp180_pressure = 0;
  if (isnan(sensorData.bmp180_altitude)) sensorData.bmp180_altitude = 0;
}

/**
 * @brief 扫描I2C总线设备
 */
void i2c_scan() {
  byte error, address;
  int nDevices = 0;

  Serial.println("开始扫描I2C设备...");
  Serial.println("地址范围: 0x01 - 0x7F");
  
  for (address = 0x01; address < 0x7F; address++) {
    Wire.beginTransmission(address);
    error = Wire.endTransmission();
    
    if (error == 0) {
      Serial.printf("发现I2C设备，地址: 0x%02X\n", address);
      nDevices++;
    } else if (error != 2 && DEBUG_MODE == 1) {
      Serial.printf("地址0x%02X通讯错误: %d\n", address, error);
    }
  }
  
  if (nDevices == 0) {
    Serial.println("未发现任何I2C设备");
  } else {
    Serial.printf("共发现 %d 个I2C设备\n", nDevices);
  }
  Serial.println();
}

/**
 * @brief 读取并计算当前流明值
 * @param samples 采样次数（默认8次）
 * @return float 计算得到的流明值
 */
float readLuxValue(int samples) {
  long sum_raw = 0;

  // 多次采样取平均值，提高稳定性
  for (int i = 0; i < samples; i++) {
    sum_raw += analogRead(LDR_ADC_PIN);
    delay(2);  // 短暂延迟避免连续读取
  }

  int raw_value = sum_raw / samples;  // 计算平均值

  // 转换为电压（考虑分压电路）
  // 假设ESP32-C5的ADC参考电压为3.3V
  float sensor_voltage = (raw_value * 3.3 / 4095.0) * VOLTAGE_DIVIDER_FACTOR;

  // 将电压转换为流明值（线性关系）
  float lux = (sensor_voltage / VOLTAGE_MAX) * LUX_MAX;

  // 确保流明值在合理范围内
  lux = constrain(lux, 0, LUX_MAX);

  return lux;
}

/**
 * @brief 将流明值转换为光线描述
 * @param lux 流明值
 * @return const char* 光线强度描述
 */
const char* convertLuxToLightLevel(float lux) {
  int level;

  // 根据流明值划分10个档位
  if (lux < 10) {
    level = 0;      // 完全黑暗
  } else if (lux < 50) {
    level = 1;      // 极弱光线
  } else if (lux < 200) {
    level = 2;      // 很弱光线
  } else if (lux < 500) {
    level = 3;      // 弱光线
  } else if (lux < 1000) {
    level = 4;      // 中等偏弱
  } else if (lux < 1500) {
    level = 5;      // 中等光线
  } else if (lux < 2500) {
    level = 6;      // 中等偏强
  } else if (lux < 3500) {
    level = 7;      // 强光线
  } else if (lux < 4500) {
    level = 8;      // 很强光线
  } else {
    level = 9;      // 刺眼光线
  }

  return LIGHT_LEVELS[level];
}

/**
 * @brief 获取光线档位数值
 * @param lux 流明值
 * @return int 光线档位（0-9）
 */
int getLightLevel(float lux) {
  if (lux < 10) return 0;
  else if (lux < 50) return 1;
  else if (lux < 200) return 2;
  else if (lux < 500) return 3;
  else if (lux < 1000) return 4;
  else if (lux < 1500) return 5;
  else if (lux < 2500) return 6;
  else if (lux < 3500) return 7;
  else if (lux < 4500) return 8;
  else return 9;
}

/**
 * @brief 读取并打印所有传感器数据
 */
void printSensorData() {
  // 格式化输出传感器数据
  Serial.println("🌿 我家大棚环境监测系统 - 1号棚");
  Serial.println("==================================================");
  Serial.printf("💡 光线: %0.0f Lux | 档位: %d | 强度: %s\n", 
                sensorData.light_lux, sensorData.light_level, sensorData.light_description);
  Serial.printf("🌡️  SHT31: 温度 %0.1f℃ | 湿度 %0.0f%%\n", 
                sensorData.sht30_temperature, sensorData.sht30_humidity);
  Serial.printf("📊 BMP180: 温度 %0.1f℃ | 气压 %0.3f kPa | 海拔 %0.1f米\n", 
                sensorData.bmp180_temperature, sensorData.bmp180_pressure / 1000, sensorData.bmp180_altitude);
  Serial.printf("🔋 电池: 电压 %0.2fV | 电量 %d%% | 状态: %s\n",
                sensorData.battery_voltage, sensorData.battery_percentage, sensorData.battery_status);
  
  // 显示休眠状态
  if (skipSleep) {
    Serial.println("⏸️ 当前状态: 休眠已取消");
  } else {
    Serial.println("💤 当前状态: 休眠已启用");
  }
  Serial.println();
}