#include "JX90614.h"

JX90614::JX90614() {
  // 构造函数可以为空
}

bool JX90614::begin(TwoWire &wirePort) {
  _wire = &wirePort;
  _i2caddr = JX90614_I2C_ADDR;
  
  _wire->begin();

  // 根据说明书(第10页)初始化传感器
  // 1. 复位CMD寄存器
  _wire->beginTransmission(_i2caddr);
  _wire->write(JX90614_REG_CMD);
  _wire->write(0x00);
  if (_wire->endTransmission() != 0) {
    return false; // 通信失败
  }
  
  delay(10); // 短暂延时

  // 2. 启动状态机 (FSM)
  _wire->beginTransmission(_i2caddr);
  _wire->write(JX90614_REG_CMD);
  _wire->write(0x08); // 启动单通道模式测量
  if (_wire->endTransmission() != 0) {
    return false; // 通信失败
  }

  return true;
}

bool JX90614::read() {
  // 读取数据就绪状态寄存器
  uint8_t dataReadyState = 0;
  _wire->beginTransmission(_i2caddr);
  _wire->write(JX90614_REG_DATA_READY_CAL);
  _wire->endTransmission(false);
  if (_wire->requestFrom(_i2caddr, (uint8_t)1) == 1) {
    dataReadyState = _wire->read();
  }

  // 检查最终温度数据是否就绪 (Temp_rdy bit 3)
  if (dataReadyState & 0b00001000) {
    // 既然有新数据了，就读取所有相关的数据
    rawTo = readThreeBytesSigned(JX90614_REG_TOBJ);
    rawTaCal = readThreeBytesSigned(JX90614_REG_TACAL);
    rawTo1Cal = readThreeBytesSigned(JX90614_REG_TO1CAL);
    rawTaRaw = readThreeBytesSigned(JX90614_REG_TARAW);
    rawTo1Raw = readThreeBytesSigned(JX90614_REG_TO1RAW);
    return true; // 表示成功读取了新数据
  }

  return false; // 表示没有新数据
}

float JX90614::getObjectTemperatureC() {
  // 根据说明书公式: T(°C) = To / 2^14
  return (float)rawTo / 16384.0;
}

float JX90614::getAmbientTemperatureC() {
  // 根据说明书公式: T(°C) = TaCal / 2^14
  return (float)rawTaCal / 16384.0;
}

double JX90614::getThermopileVoltage() {
  // 根据说明书公式: Vtp(mV) = To1Cal / 2^19 (在±16mV挡位下)
  return (double)rawTo1Cal / 524288.0;
}

long JX90614::getRawObject() {
  return rawTo;
}

long JX90614::getRawAmbient() {
  return rawTaCal;
}

long JX90614::getRawThermopileCal() {
  return rawTo1Cal;
}

long JX90614::getRawAmbientADC() {
  return rawTaRaw;
}

long JX90614::getRawThermopileADC() {
  return rawTo1Raw;
}

long JX90614::readThreeBytesSigned(uint8_t startRegister) {
  long value = 0;
  uint8_t b1, b2, b3;

  _wire->beginTransmission(_i2caddr);
  _wire->write(startRegister);
  _wire->endTransmission(false);

  if (_wire->requestFrom(_i2caddr, (uint8_t)3) == 3) {
    b1 = _wire->read(); // 最高字节 (MSB)
    b2 = _wire->read(); // 中间字节 (CSB)
    b3 = _wire->read(); // 最低字节 (LSB)
    
    value = (long)b1 << 16 | (long)b2 << 8 | (long)b3;

    // 如果是负数，进行符号位扩展
    if (value & 0x00800000) {
      value |= 0xFF000000;
    }
  } else {
    return 0; // 错误状态
  }
  return value;
}