#include "Arduino.h"
#include <Wire.h>
#include "SimpleFOC.h"
#include <WiFi.h>
#include <ESPAsyncWebServer.h>

/*----定义两个电机的磁编码器连接引脚----*/
#define M0_I2C_SDA 21
#define M0_I2C_SCL 22
#define M1_I2C_SDA 19
#define M1_I2C_SCL 18

// WiFi 配置
const char* ssid = "sxs";       // 替换为您的WiFi名称
const char* password = "smj080823"; // 替换为您的WiFi密码

AsyncWebServer server(80);

// 电机控制参数
float base_speed = 30.0;        // 基础速度（扭矩值），可通过滑块调节
float turn_speed = 20.0;        // 转向速度（扭矩值），可通过滑块调节
float left_motor_cmd = 0.0;
float right_motor_cmd = 0.0;

// 电机对象
MagneticSensorI2C sensor_0 = MagneticSensorI2C(AS5600_I2C);
TwoWire I2C_0 = TwoWire(0);
MagneticSensorI2C sensor_1 = MagneticSensorI2C(AS5600_I2C);
TwoWire I2C_1 = TwoWire(1);

BLDCMotor motor_0 = BLDCMotor(7);
BLDCDriver3PWM driver_0 = BLDCDriver3PWM(25, 26, 27, 9);
BLDCMotor motor_1 = BLDCMotor(7);
BLDCDriver3PWM driver_1 = BLDCDriver3PWM(5, 13, 10, 2);

// HTML 网页内容（与之前相同，仅修改提示文字）
const char index_html[] PROGMEM = R"rawliteral(
<!DOCTYPE html>
<html lang="zh">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, user-scalable=no">
    <title>两轮差速小车遥控</title>
    <style>
        body { font-family: Arial, sans-serif; text-align: center; margin: 20px; }
        .btn {
            width: 120px; height: 120px; font-size: 24px; margin: 10px; border-radius: 10px;
            border: none; background-color: #4CAF50; color: white; touch-action: manipulation;
            /* 禁用长按选中和弹出菜单 */
            -webkit-touch-callout: none;
            -webkit-user-select: none;
            -khtml-user-select: none;
            -moz-user-select: none;
            -ms-user-select: none;
            user-select: none;
            -webkit-tap-highlight-color: transparent;
        }
        .btn:active { background-color: #45a049; }
        .container { display: flex; flex-wrap: wrap; justify-content: center; }
        .slider-container { margin: 20px; width: 80%; max-width: 400px; margin-left: auto; margin-right: auto; }
        .slider { width: 100%; }
        .value { font-size: 20px; margin: 10px; }
    </style>
</head>
<body>
    <h1>两轮差速小车遥控器</h1>
    <div class="container">
        <button class="btn" id="forward" ontouchstart="sendAction('f', event)" ontouchend="sendAction('s', event)" onmousedown="sendAction('f', event)" onmouseup="sendAction('s', event)">前进</button>
        <button class="btn" id="backward" ontouchstart="sendAction('b', event)" ontouchend="sendAction('s', event)" onmousedown="sendAction('b', event)" onmouseup="sendAction('s', event)">后退</button>
        <button class="btn" id="left" ontouchstart="sendAction('l', event)" ontouchend="sendAction('s', event)" onmousedown="sendAction('l', event)" onmouseup="sendAction('s', event)">左转</button>
        <button class="btn" id="right" ontouchstart="sendAction('r', event)" ontouchend="sendAction('s', event)" onmousedown="sendAction('r', event)" onmouseup="sendAction('s', event)">右转</button>
        <button class="btn" id="stop" onclick="sendAction('s', event)">停止</button>
    </div>

    <div class="slider-container">
        <h3>基础速度: <span id="baseSpeedValue">30</span></h3>
        <input type="range" min="0" max="100" step="1" value="30" class="slider" id="baseSpeedSlider" oninput="updateBaseSpeed(this.value)">
    </div>
    <div class="slider-container">
        <h3>转向速度: <span id="turnSpeedValue">20</span></h3>
        <input type="range" min="0" max="100" step="1" value="20" class="slider" id="turnSpeedSlider" oninput="updateTurnSpeed(this.value)">
    </div>

    <script>
        function sendAction(cmd, event) {
            if (event) {
                event.preventDefault();  // 阻止默认行为（长按菜单、文本选择等）
            }
            fetch(`/action?cmd=${cmd}`)
                .then(response => console.log(`Action ${cmd} sent`))
                .catch(error => console.error('Error:', error));
        }

        function updateBaseSpeed(val) {
            document.getElementById('baseSpeedValue').innerText = val;
            fetch(`/set?param=base_speed&value=${val}`)
                .then(response => console.log('Base speed set to ' + val))
                .catch(error => console.error('Error:', error));
        }

        function updateTurnSpeed(val) {
            document.getElementById('turnSpeedValue').innerText = val;
            fetch(`/set?param=turn_speed&value=${val}`)
                .then(response => console.log('Turn speed set to ' + val))
                .catch(error => console.error('Error:', error));
        }
    </script>
</body>
</html>
)rawliteral";

void sensor_init() {
    I2C_0.begin(M0_I2C_SDA, M0_I2C_SCL, 100000);
    sensor_0.init(&I2C_0);
    I2C_1.begin(M1_I2C_SDA, M1_I2C_SCL, 100000);
    sensor_1.init(&I2C_1);
}

void motor_init() {
    // 电机0初始化
    motor_0.linkSensor(&sensor_0);
    driver_0.voltage_power_supply = 12;
    driver_0.voltage_limit = 6;
    driver_0.init();
    motor_0.phase_resistance = 2.9;
    motor_0.torque_controller = TorqueControlType::voltage;
    motor_0.linkDriver(&driver_0);
    motor_0.foc_modulation = FOCModulationType::SpaceVectorPWM;
    motor_0.controller = MotionControlType::torque;  // 保持扭矩模式
    motor_0.PID_velocity.P = 0.108;
    motor_0.PID_velocity.I = 0.001;
    motor_0.LPF_velocity.Tf = 0.01f;
    motor_0.P_angle.P = 30;
    motor_0.velocity_limit = 20;
    motor_0.useMonitoring(Serial);
    motor_0.init();
    motor_0.sensor_direction = Direction::CCW;
    motor_0.initFOC();

    // 电机1初始化
    motor_1.linkSensor(&sensor_1);
    driver_1.voltage_power_supply = 12;
    driver_1.voltage_limit = 6;
    driver_1.init();
    motor_1.phase_resistance = 2.9;
    motor_1.torque_controller = TorqueControlType::voltage;
    motor_1.linkDriver(&driver_1);
    motor_1.foc_modulation = FOCModulationType::SpaceVectorPWM;
    motor_1.controller = MotionControlType::torque;
    motor_1.PID_velocity.P = 0.105;
    motor_1.PID_velocity.I = 0.001;
    motor_1.LPF_velocity.Tf = 0.01f;
    motor_1.P_angle.P = 30;
    motor_1.velocity_limit = 20;
    motor_1.useMonitoring(Serial);
    motor_1.init();
    motor_1.sensor_direction = Direction::CCW;
    motor_1.initFOC();
}

// 根据命令更新电机目标值
void updateMotorCommands(char cmd) {
    switch (cmd) {
        case 's': // 停止
            left_motor_cmd = 0;
            right_motor_cmd = 0;
            break;
        case 'f': // 前进
            left_motor_cmd = base_speed;
            right_motor_cmd = base_speed;
            break;
        case 'b': // 后退
            left_motor_cmd = -base_speed;
            right_motor_cmd = -base_speed;
            break;
        case 'l': // 左转（左轮慢/右轮快，此处用差速方式：左轮反转，右轮正转）
            left_motor_cmd = -turn_speed;
            right_motor_cmd = turn_speed;
            break;
        case 'r': // 右转
            left_motor_cmd = turn_speed;
            right_motor_cmd = -turn_speed;
            break;
        default:
            break;
    }
}

// Web请求处理函数
void handleAction(AsyncWebServerRequest *request) {
    if (request->hasParam("cmd")) {
        String cmd = request->getParam("cmd")->value();
        if (cmd.length() == 1) {
            updateMotorCommands(cmd.charAt(0));
        }
    }
    request->send(200, "text/plain", "OK");
}

void handleSet(AsyncWebServerRequest *request) {
    if (request->hasParam("param") && request->hasParam("value")) {
        String param = request->getParam("param")->value();
        float val = request->getParam("value")->value().toFloat();
        if (param == "base_speed") {
            base_speed = val;
        } else if (param == "turn_speed") {
            turn_speed = val;
        }
    }
    request->send(200, "text/plain", "OK");
}

void setup() {
    Serial.begin(115200);
    sensor_init();
    motor_init();

    Serial.println("Motor ready!");

    // 连接WiFi
    WiFi.begin(ssid, password);
    Serial.print("Connecting to WiFi");
    while (WiFi.status() != WL_CONNECTED) {
        delay(500);
        Serial.print(".");
    }
    Serial.println();
    Serial.print("Connected! IP address: ");
    Serial.println(WiFi.localIP());

    // 配置Web服务器路由
    server.on("/", HTTP_GET, [](AsyncWebServerRequest *request){
        request->send_P(200, "text/html", index_html);
    });
    server.on("/action", HTTP_GET, handleAction);
    server.on("/set", HTTP_GET, handleSet);

    server.begin();
    Serial.println("Web server started!");
}

void loop() {
    // 更新传感器和电机FOC
    sensor_0.update();
    sensor_1.update();
    motor_0.loopFOC();
    motor_1.loopFOC();

    // 应用电机指令（扭矩模式）
    motor_0.move(-left_motor_cmd);   // 根据原代码符号，左电机取反
    motor_1.move(right_motor_cmd);   // 右电机不变

    // 可选：打印当前指令
    // Serial.print("L: "); Serial.print(left_motor_cmd);
    // Serial.print("  R: "); Serial.println(right_motor_cmd);

    delay(10); // 小延时避免过度占用CPU
}